
#ifndef XBOXFLASH_H
#define XBOXFLASH_H

#include <stdio.h>
#include <stdarg.h>

#define	FLASH_STATUS_DQ7			0x80
#define FLASH_STATUS_TOGGLE			0x40
#define FLASH_STATUS_ERROR			0x20
#define FLASH_STATUS_ERASE_TIMER	0x08
#define FLASH_STATUS_TOGGLE_ALT		0x04

#define FLASH_BASE_ADDRESS			0xff000000
#define	FLASH_UNLOCK_ADDR_1			(FLASH_BASE_ADDRESS+0x0555)
#define	FLASH_UNLOCK_ADDR_2			(FLASH_BASE_ADDRESS+0x02aa)

#define FLASH_UNLOCK_DATA_1			0xaa
#define FLASH_UNLOCK_DATA_2			0x55

#define FLASH_COMMAND_RESET			0xf0
#define FLASH_COMMAND_AUTOSELECT	0x90
#define FLASH_COMMAND_PROGRAM		0xa0
#define FLASH_COMMAND_UNLOCK_BYPASS	0x20
#define FLASH_COMMAND_ERASE			0x80

#define FLASH_COMMAND_ERASE_BLOCK	0x30
#define FLASH_COMMAND_ERASE_ALL		0x10

#define FLASH_BLOCK_COUNT			0x10
#define FLASH_BLOCK_MASK			(FLASH_BLOCK_COUNT-1)

typedef struct fci_s {
	unsigned char mfct;
	unsigned char devc;
	WCHAR *text;
	unsigned long size;
	struct fci_s *next;
} fci_t;

class CXBoxFlash
{
	fci_t *flashinfo;
	void (*UpdateStatus)(WCHAR *line1, WCHAR *line2, unsigned long addr);

public:
	CXBoxFlash()
	{
		flashinfo=NULL;
	}
	
	~CXBoxFlash(void)
	{
	}

	fci_t* AddFCI(BYTE manuf, BYTE code, WCHAR* text, unsigned long size)
	{
		fci_t *cur = flashinfo;

		if(cur==NULL) {
			flashinfo = (fci_t*)malloc(sizeof(fci_t));
			cur=flashinfo;
		} else {
			while(cur->next!=NULL) cur=cur->next;
			cur->next = (fci_t*)malloc(sizeof(fci_t));
			cur=cur->next;
		}
		memset(cur, 0, sizeof(fci_t));

		cur->mfct = manuf;
		cur->devc = code;
		cur->text = lstrdupW(text);
		cur->size = size;

		return cur;
	}

	fci_t* FindFCI(BYTE manuf, BYTE code)
	{
		fci_t *cur = flashinfo;

		while(cur!=NULL) {
			if((cur->mfct==manuf) && (cur->devc==code)) return cur;
			cur=cur->next;
		}
		return NULL;
	}

	inline void Write(DWORD address,BYTE data)
	{
		volatile BYTE *ptr=(BYTE*)address;
		*ptr=data;
	}

	inline BYTE Read(DWORD address)
	{
		volatile BYTE *ptr=(BYTE*)address;
		return *ptr;
	}

	void SetReadMode(void)
	{
		// 	Unlock stage 1
		Write(FLASH_UNLOCK_ADDR_1,FLASH_UNLOCK_DATA_1); 
		Write(FLASH_UNLOCK_ADDR_2,FLASH_UNLOCK_DATA_2);
		// Issue the reset
		Write(FLASH_UNLOCK_ADDR_1,FLASH_COMMAND_RESET);

		// Leave it in a read mode to avoid any buss contention issues
		BYTE dummy=Read(FLASH_BASE_ADDRESS);
	}

	fci_t* CheckID(void)
	{
		BYTE manuf,code;

		// 	Unlock stage 1
		Write(FLASH_UNLOCK_ADDR_1,FLASH_UNLOCK_DATA_1);
		Write(FLASH_UNLOCK_ADDR_2,FLASH_UNLOCK_DATA_2);
		// Issue the autroselect
		Write(FLASH_UNLOCK_ADDR_1,FLASH_COMMAND_AUTOSELECT);

		manuf=Read(FLASH_BASE_ADDRESS);
		code=Read(FLASH_BASE_ADDRESS+1);
		
		// All done
		SetReadMode();

		return FindFCI(manuf,code);
	}

	bool EraseBlock(int block)
	{
		bool retval;

		// 	Unlock stage 1
		Write(FLASH_UNLOCK_ADDR_1,FLASH_UNLOCK_DATA_1);
		Write(FLASH_UNLOCK_ADDR_2,FLASH_UNLOCK_DATA_2);
		// Issue the erase
		Write(FLASH_UNLOCK_ADDR_1,FLASH_COMMAND_ERASE);
		// Unlock stage 2
		Write(FLASH_UNLOCK_ADDR_1,FLASH_UNLOCK_DATA_1);
		Write(FLASH_UNLOCK_ADDR_2,FLASH_UNLOCK_DATA_2);
		// Now set the block
		Write((FLASH_UNLOCK_ADDR_1+(block&FLASH_BLOCK_MASK)),FLASH_COMMAND_ERASE_BLOCK);
		
		// Now poll for a result
		retval=WaitOnToggle();
		
		// All done		
		SetReadMode();

		return retval;
	}

	bool EraseDevice(void)
	{
		bool retval;

		// 	Unlock stage 1
		Write(FLASH_UNLOCK_ADDR_1,FLASH_UNLOCK_DATA_1);
		Write(FLASH_UNLOCK_ADDR_2,FLASH_UNLOCK_DATA_2);
		// Issue the erase
		Write(FLASH_UNLOCK_ADDR_1,FLASH_COMMAND_ERASE);
		// Unlock stage 2
		Write(FLASH_UNLOCK_ADDR_1,FLASH_UNLOCK_DATA_1);
		Write(FLASH_UNLOCK_ADDR_2,FLASH_UNLOCK_DATA_2);
		// Now set the block
		Write(FLASH_UNLOCK_ADDR_1,FLASH_COMMAND_ERASE_ALL);
		
		// Now poll for a result
		retval=WaitOnToggle();

		// All done
		SetReadMode();

		// Check it all 0xFF
		for(DWORD address=FLASH_BASE_ADDRESS;address<(FLASH_BASE_ADDRESS+0x100000);address++)
		{
			if(Read(address)!=0xff)
			{
				WCHAR tmpW[128];
				wsprintfW(tmpW, L"Unable to erase 0x%08x", address);
				UpdateStatus(L"Erase Failure:", tmpW, address);
				return false;
			}
		}

		return retval;
	}

	int ProgramDevice(char *filename)
	{
		FILE *fp;
		BYTE data;
		volatile BYTE dummy;
		DWORD twiddle=0;
		DWORD address=FLASH_BASE_ADDRESS;
		WCHAR tmpW[256];

		if((fp=fopen(filename,"rb"))==NULL) return false;

		while(fread(&data,1,sizeof(BYTE),fp)==1)
		{
			// Check address bound
			if(address>=FLASH_BASE_ADDRESS+0x100000)
			{
				UpdateStatus(L"Programming Failure:", L"Incorrect Filesize", address);
				fclose(fp);
				SetReadMode();
				return -1;
			}

			// 	Unlock stage 1
			Write(FLASH_UNLOCK_ADDR_1,FLASH_UNLOCK_DATA_1);
			Write(FLASH_UNLOCK_ADDR_2,FLASH_UNLOCK_DATA_2);
			// Issue the program command
			Write(FLASH_UNLOCK_ADDR_1,FLASH_COMMAND_PROGRAM);
			// Program Byte
			Write(address,data);

			// Do the Data polling test
			while(1)
			{
				dummy=Read(address);

				if((data&FLASH_STATUS_DQ7)==(dummy&FLASH_STATUS_DQ7)) break;

				if(dummy&FLASH_STATUS_ERROR)
				{
					dummy=Read(address);
					if((data&FLASH_STATUS_DQ7)==(dummy&FLASH_STATUS_DQ7))
					{
						break;
					}
					else
					{
						wsprintfW(tmpW, L"Write failure at 0x%08x", address);
						UpdateStatus(L"Programming Failure:", tmpW, address);
						fclose(fp);
						SetReadMode();
						return -2;
					}
				}
			}
			Write(FLASH_UNLOCK_ADDR_1,FLASH_COMMAND_RESET);

			dummy=Read(address);
			// Verify the written byte
			if(dummy!=data)
			{
				wsprintfW(tmpW, L"Verify failure at 0x%08x, Wrote 0x%02x, Read 0x%02x", address, data, dummy);
				UpdateStatus(L"Programming Failure:", tmpW, address);
				fclose(fp);
				SetReadMode();
				return -3;
			}

			// Next byte
			address++;

			// User information
			if((address&0xffff)==0x0000)
			{
				wsprintfW(tmpW, L"Wrote block %02d",(address>>16)&0xff);
				UpdateStatus(L"Writing...", tmpW, address);
			}
		}

		wsprintfW(tmpW, L"Wrote %i bytes", address && 0xffffff);
		UpdateStatus(L"Verifying...", tmpW, address);

		// Verify written code
		int count=0;
		fseek(fp,0,SEEK_SET);
		address=FLASH_BASE_ADDRESS;
		while(fread(&data,1,sizeof(BYTE),fp)==1)
		{
			if(data!=Read(address))
			{
				wsprintfW(tmpW, L"Verify failure at 0x%08x, Wrote 0x%02x, Read 0x%02x", address, data, Read(address));
				UpdateStatus(L"Programming Failure:", tmpW, address);
				if(count++>8) return -3;
			}
			address++;
		}

		fclose(fp);

		if(count) {
			UpdateStatus(L"Programming Failure:", L"One or more bytes did not write propperly.", FLASH_BASE_ADDRESS);
			return -3;
		} else {
			UpdateStatus(L"Programming Complete:", L"Flash Programming Successful.", address-1);
			return 0;
		}
	}

	bool WaitOnToggle(void)
	{
		BYTE last,current;
		last=Read(FLASH_BASE_ADDRESS);
		while(1)
		{
			// We wait for the toggle bit to stop toggling
			current=Read(FLASH_BASE_ADDRESS);
			// Check for an end to toggling
			if((last&FLASH_STATUS_TOGGLE)==(current&FLASH_STATUS_TOGGLE)) break;
			last=current;

			// We're still in command mode so its OK to check for Error condition
			if(current&FLASH_STATUS_ERROR)
			{
				last=Read(FLASH_BASE_ADDRESS);
				current=Read(FLASH_BASE_ADDRESS);
				if((last&FLASH_STATUS_TOGGLE)==(current&FLASH_STATUS_TOGGLE)) break; else return false;
			}
		}
		return true;
	}

};

#endif